'use strict';

/*
 * ==========================================================================
 *  INIT
 * ==========================================================================
 */

/*
 * Load gulp.
 */
var gulp = require('gulp');

/*
 * Autoload gulp plugins.
 */
var plugins = require('gulp-load-plugins')();

/*
 * Time gulp tasks execution time.
 */
require('gulp-stats')(gulp);

/*
 * Parse command-line for options.
 */
var argv = require('yargs').argv;

/*
 * ==========================================================================
 *  CONFIGURATION
 * ==========================================================================
 */

var fs = require('fs');
var Q = require('q');

var bower_options = {
    'directory': getDirectoryFromBowerRc() || 'bower_components'
}

var project = (function() {
    var scriptsSrcPath = 'assets/js',
        stylesSrcPath = 'assets/css';
    return {
        'src': {
            'css': {
                'directory': stylesSrcPath,
                'files': [
                    bower_options.directory + '/jquery-ui/themes/smoothness/jquery-ui.css',
                    bower_options.directory + '/jquery-ui/themes/smoothness/jquery.ui.theme.css',
                    bower_options.directory + '/ionicons/css/ionicons.min.css',
                    bower_options.directory + '/fancybox/source/jquery.fancybox.css',
                ]
            },
            'js': {
                'directory': scriptsSrcPath,
                'files': [
                    bower_options.directory + '/jsgettext/lib/Gettext.js',
                    bower_options.directory + '/jquery-ui/ui/minified/jquery-ui.min.js',
                    bower_options.directory + '/jqueryui-touch-punch/jquery.ui.touch-punch.min.js',
                    bower_options.directory + '/twig.js/twig.min.js',
                    bower_options.directory + '/slick-carousel/slick/slick.min.js',
                    bower_options.directory + '/jquery-validation/dist/jquery.validate.min.js',
                    bower_options.directory + '/dropzone/dist/min/dropzone.min.js',
                    bower_options.directory + '/fancybox/source/jquery.fancybox.js',
                    bower_options.directory + '/slideout.js/dist/slideout.min.js',
                    bower_options.directory + '/masonry/dist/masonry.pkgd.min.js',
                    bower_options.directory + '/imagesloaded/imagesloaded.pkgd.min.js',
                    bower_options.directory + '/jquery-text-counter/textcounter.min.js',
                    bower_options.directory + '/bootstrap-sass/assets/javascripts/bootstrap.min.js',
                    bower_options.directory + '/jquery.serializeJSON/jquery.serializejson.js',
                    bower_options.directory + '/underscore/underscore.js',
                    scriptsSrcPath + '/messenger.js',
                    scriptsSrcPath + '/osclass.date.js',
                    scriptsSrcPath + '/item.uploader.js',
                    scriptsSrcPath + '/item.watchlist.js',
                    scriptsSrcPath + '/item.post.js',
                    scriptsSrcPath + '/search.js',
                    scriptsSrcPath + '/search.ajax.js',
                    scriptsSrcPath + '/main.js',
                ]
            },
        },
        'dest': {
            'css': {
                'directory': stylesSrcPath + '/dist'
            },
            'js': {
                'directory': scriptsSrcPath + '/dist',
                'filename': 'main.js'
            }
        }
    };
})();

/*
 * ==========================================================================
 *  TASKS
 * ==========================================================================
 */

/*
 * Sass.
 *
 * Compile theme variant scss files.
 */
gulp.task('sass', function() {
    var sassFiles = [
        project.src.css.directory + '/theme1.scss',
        project.src.css.directory + '/theme2.scss'
    ];
    if (argv.theme) {
        sassFiles = project.src.css.directory + '/' + argv.theme + '.scss';
    }

    return gulp
        // Find all `.scss` files from the `stylesheets/` folder
        .src(sassFiles)
        .pipe(plugins.plumber())
        // Run Sass on those files
        .pipe(plugins.sass({
            includePaths: bower_options.directory,
            precision: 10,
        }))
        // Write the resulting CSS in the output folder
        .pipe(gulp.dest(project.dest.css.directory))
    ;
});

/*
 * Minify CSS.
 *
 * Minify theme variant ${dest}/*.css to a ${dest}.min.css
 */
gulp.task('minify-css', ['sass'], function() {
    var cssFiles = ['theme1', 'theme2'];
    if (argv.theme) {
        cssFiles = [argv.theme];
    }

    var promises = cssFiles.map(function (variant) {
        var deferred = Q.defer();

        gulp.src(project.src.css.files.concat([project.dest.css.directory + '/' + variant + '.css']))
            .pipe(plugins.plumber())
            // Concat and minify css files into one.
            .pipe(
                plugins.cleanCss(
                    {
                        debug: true,
                        rebase: true,
                        relativeTo: project.dest.css.directory,
                        target: project.dest.css.directory
                    },
                    function(details) {
                        plugins.util.log(plugins.util.colors.green('- ' + details.name + ': ' + formatBytes(details.stats.originalSize) + ' -> ' + formatBytes(details.stats.minifiedSize)));
                    }
                )
            ).pipe(plugins.concat(variant + '.min.css'))
            //).pipe(plugins.rename({ extname: '.min.css' }))
            // .pipe(sourcemaps.write('.'))
            .pipe(gulp.dest(project.dest.css.directory))
        ;

        return deferred.promise;
    });
    Q.all(promises);
});

/*
 * Concat.
 *
 * Concat javacsript files to one.
 */
gulp.task('scripts-concat', function() {
    return gulp.src(project.src.js.files).pipe(plugins.plumber())
        // Concat all files into one main.js
        .pipe(plugins.concat(project.dest.js.filename))
        // Write to dest/ directory
        .pipe(gulp.dest(project.dest.js.directory))
    ;
});

/*
 * Uglify.
 *
 * Minify javacsript files to dist/
 */
gulp.task('scripts-uglify', ['scripts-concat'], function() {
    return gulp.src(project.dest.js.directory + '/main.js').pipe(plugins.plumber())
        // Uglify javascript
        .pipe(plugins.uglify({
            compress: true,
            mangle: true,
            preserveComments: 'some',
            drop_console: true,
        }))
        // Rename to *.min.js
        .pipe(plugins.rename({ extname: '.min.js' }))
        // Write to dest/ directory
        .pipe(gulp.dest(project.dest.js.directory))
    ;
});

/*
 * Build css and js.
 */
gulp.task('build', ['minify-css', 'scripts-uglify'], function() {

});


/*
 * Watch.
 *
 * Monitor all files and trigger tasks.
 */
gulp.task('watch', ['minify-css', 'scripts-uglify'], function() {
    var options = { usePolling: true };

    plugins
        // Watch the input folder for change,
        .watch(
            [
                project.src.css.directory + '/**/*.scss',
                bower_options.directory + '/cssr/src/**/*.scss'
            ],
            options,
            function (events, done) {
                gulp.start(['sass', 'minify-css']);
            }
        )
        // Display the file name
        .on('change', function(file) {
            plugins.util.log(plugins.util.colors.yellow('CSS changed' + ' (' + file + ')'));
        })
    ;

    plugins
        // Watch the input folder for change,
        .watch(
            [
                project.src.js.directory + '/**/*.js',
                '!' + project.dest.js.directory + '/*.js'
            ],
            options,
            function (events, done) {
                gulp.start(['scripts-concat', 'scripts-uglify']);
            }
        )
        // Display the file name
        .on('change', function(file) {
            plugins.util.log(plugins.util.colors.yellow('JS changed' + ' (' + file + ')'));
        })
    ;
});

/*
 * Default task.
 */
gulp.task('default', ['watch']);

/*
 * ==========================================================================
 *  UTILS
 * ==========================================================================
 */

// https://github.com/zont/gulp-bower
/**
 * Detect .bowerrc file and read directory from file
 *
 * @param {string} cwd current working directory
 * @returns {string} found directory or empty string
 */

function getDirectoryFromBowerRc(cwd) {
    var bowerrc = './.bowerrc';
    if (!fs.existsSync(bowerrc)) {
        return '';
    }
    var bower_config = {};
    try {
        bower_config = JSON.parse(fs.readFileSync(bowerrc));
        return bower_config.directory;
    } catch (err) {
        return '';
    }
}

// http://stackoverflow.com/questions/15900485/correct-way-to-convert-size-in-bytes-to-kb-mb-gb-in-javascript
function formatBytes(bytes, decimals) {
    if (bytes == 0) return '0 Byte';
    var k = 1000; // or 1024 for binary
    var dm = decimals + 1 || 3;
    var sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB', 'PB', 'EB', 'ZB', 'YB'];
    var i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
}
