<?php

/**
 * Returns the path for a themekitchen resource within 'uploads/' folder.
 * @return a string.
 * @since 1.00
 */
function mtk_uploads_path($file = null)
{
    $path = osc_uploads_path() . Madhouse_ThemeKitchen_Package::getThemeSlug();
    if (!is_null($file)) {
        $path .= "/" . $file;
    }
    return $path;
}

/**
 * Theme settings URL.
 * @return URL (string)
 * @since 1.00
 */
function mtk_settings_url($submoduleId = "general")
{
    return osc_admin_render_theme_url(
        Madhouse_ThemeKitchen_Package::getRelativePath("views/admin/settings.php")
        . "&submodule_id=" . $submoduleId
    );
}

/**
 * Return the url to the online documentation
 * @return string The documentation url
 */
function mtk_help_url()
{
    return "http://wearemadhouse.wordpress.com/" . osc_current_web_theme() . "-documentation";
}

/**
 * Theme settings post URL.
 * @return URL (string)
 * @since 1.00
 */
function mtk_settings_post_url()
{
    return osc_admin_render_theme_url(Madhouse_ThemeKitchen_Package::getRelativePath("views/admin/settings_post.php"));
}

/**
 * Delete resource URL.
 *
 * @return URL (string)
 *
 * @since 1.1.0
 */
function mtk_settings_delete_resource_post_url($section, $name, $locale = "")
{
    $query = http_build_query(
        array(
            "resource_section" => $section,
            "resource_name" => $name,
            "resource_locale" => $locale,
        )
    );
    return osc_admin_render_theme_url(Madhouse_ThemeKitchen_Package::getRelativePath("views/admin/resource_delete_post.php"))
        . "&" . $query;
}

/**
 * Is the current the theme settings URL ?
 * @return true/false (boolean)
 * @since 1.00
 */
function mtk_is_settings_page($submoduleId = null)
{
    $relativePath = Madhouse_ThemeKitchen_Package::getRelativePath("views/admin/settings.php");
    ;

    return (
        Params::getParam("page") === "appearance"
        && Params::getParam("action") === "render"
        && Params::getParam("file") === $relativePath
        && (is_null($submoduleId) || (!is_null($submoduleId) && Params::getParam("submodule_id") === $submoduleId))
    );
}

/**
 * Is the current the theme settings post URL ?
 * @return true/false (boolean)
 * @since 1.00
 */
function mtk_is_settings_post_page()
{
    return (
        Params::getParam("page") === "appearance"
        && Params::getParam("action") === "render"
        && Params::getParam("file") === Madhouse_ThemeKitchen_Package::getRelativePath("views/admin/settings_post.php")
    );
}

/**
 * Is the current page the theme settings delete URL ?
 *
 * @return boolean
 *
 * @since  1.1.0
 */
function mtk_settings_delete_resource_post_page()
{
    return (
        Params::getParam("page") === "appearance"
        && Params::getParam("action") === "render"
        && Params::getParam("file") === Madhouse_ThemeKitchen_Package::getRelativePath("views/admin/resource_delete_post.php")
    );
}

/**
 * URL for the logo.
 * @since 1.00
 */
function mtk_logo_url($locale = null, $fallback = true)
{
    if (is_null($locale) && $fallback === true) {
        $locale = osc_current_user_locale();
    }
    return mtk_resource_url("$.general", "s_header_logo", $locale, $fallback);
}

/**
 * URL for the variant logo.
 * @since 1.00
 */
function mtk_logo_variant_url($locale = null, $fallback = true)
{
    if (is_null($locale) && $fallback === true) {
        $locale = osc_current_user_locale();
    }
    return mtk_resource_url("$.general", "s_logo_variant", $locale, $fallback);
}

/**
 * URL for the favicon.
 * @since 1.00
 */
function mtk_favicon_url($locale = null, $fallback = true)
{
    if (is_null($locale) && $fallback === true) {
        $locale = osc_current_user_locale();
    }
    return mtk_resource_url("$.general", "s_favicon", $locale, $fallback);
}

/**
 * URL for the no-photo.
 * @since 1.00
 */
function mtk_no_photo_url($format = "preview")
{
    return mtk_resource_url("$.layout", "s_no_photo", "", true, $format);
}

/**
 *
 * @since 1.00
 */
function mtk_field($section, $field = null, $locale = "", $fallback = true)
{

    // View service.
    $view = View::newInstance();

    // Compute current settings key.
    $settingsKey = "mdh_" . Madhouse_ThemeKitchen_Package::getThemeSlug() . "_settings";

    // Has the settings been already exported ?
    if (!$view->_exists($settingsKey)) {
        // Settings service.
        $serviceSettings = Madhouse_ThemeKitchen_Services_SettingsService::newInstance();
        $currentSettings = $serviceSettings->findSettingsBy();

        // Export to view.
        $view->_exportVariableToView($settingsKey, $currentSettings);
    }

    // Get the requested field from current settings.
    return __get($settingsKey)->get($section, $field, $locale, $fallback);
}

function mtk_resource_url($section, $field = null, $locale = "", $fallback = true, $format = "")
{
    $resource = mtk_field($section, $field, $locale, $fallback);
    if (!empty($resource)) {
        $relativePath = "";
        if (!is_array($resource)) {
            $relativePath = $resource;
        } else {
            if ($format !== "") {
                $format = "_" . $format;
            }
            $relativePath = $resource["s_path"] . '/' . $resource["s_name"] . $format . "." .  $resource["s_ext"];
        }

        if (!file_exists(ABS_PATH . '/' . $relativePath)) {
            return "";
        }

        return osc_base_url() . $relativePath;
    }
    return "";
}

/**
 * Get all the blocks definition.
 *
 * @return array
 */
function mtk_config_blocks()
{
    // View service.
    $view = View::newInstance();

    // Compute current settings key.
    $configKey = "mdh_" . Madhouse_ThemeKitchen_Package::getThemeSlug() . "_config_blocks";

    // Has the config been already exported ?
    if (!$view->_exists($configKey)) {
        $configSettings = Madhouse_ThemeKitchen_Services_StaticConfigurationService::newInstance();
        $blocks = $configSettings->findBlocks();

        // Export to view.
        $view->_exportVariableToView($configKey, $blocks);
    }

    // Get the blocks configuration.
    return __get($configKey);
}

function mtk_block_path($blockSlug)
{
    $block = Madhouse_ThemeKitchen_Services_StaticConfigurationService::newInstance()->findBlock(
        $blockSlug,
        mtk_config_blocks()
    );

    if ($block !== null) {
        return $block["s_path"];
    }
    return "";
}

/**
 * Redirection for dashboard if settings requires it.
 *
 * @hook init
 *
 * @return void
 */
function mtk_user_dashboard_redirect()
{
    if (osc_is_user_dashboard()) {
        if (mtk_field("$.layout", "i_dashboard_activation") != 0) {
            // Get settings for dashboard.
            $redirectURL = "";
            switch (mtk_field("$.layout", "i_dashboard_activation")) {
                case 1:
                    // Redirect to user's list of items.
                    $redirectURL = osc_user_list_items_url();
                    break;
                case 2:
                default:
                    // Redirect to user profile (edit my profile).
                    $redirectURL = osc_user_profile_url();
                    break;
            }

            // Redirect.
            osc_redirect_to($redirectURL, 301);
        }
    }
}

function mtk_render_block($blockSlug)
{
    // Resolve path to the block.
    $blockPath = mtk_block_path($blockSlug);

    // Render it.
    if ($blockPath) {
        require mtk_block_path($blockSlug);
    }
}

function mtk_render_twig($template, $data = null)
{
    if (is_null($data)) {
        $data = array();
    }

    if (__get("mtk_templating_engine") === "") {
        $loader = new Twig_Loader_Filesystem();
        $loader->addPath(WebThemes::newInstance()->getCurrentThemePath(), 'web');
        $loader->addPath(dirname(dirname(__FILE__)).'/views/admin/parts', 'admin');
        $loader->addPath(WebThemes::newInstance()->getCurrentThemePath());
        $loader->addPath(dirname(dirname(__FILE__)).'/views/admin/parts');

        $twig = new Twig_Environment($loader);

        $twig->addExtension(new Twig_Extensions_Extension_Text());

        $twig->addFilter(new Twig_SimpleFilter("better_merge", function ($first, $second) {
            return array_replace_recursive($first, $second);
        }));
        $twig->addFilter(new Twig_SimpleFilter("hookable_to", function ($customBlocks, $hook, $group) {
            return array_filter(
                $customBlocks,
                function ($customBlock) use ($hook, $group) {
                    return (
                        empty($customBlock["widget_hooks"])
                        || in_array($hook, $customBlock["widget_hooks"])
                        || in_array($group, $customBlock["widget_hooks"])
                    );
                }
            );
        }));
        $twig->addFunction(new Twig_SimpleFunction("mtk_field", function ($section, $name, $locale = "", $fallback = false) {
            return mtk_field($section, $name, $locale, $fallback);
        }));
        $twig->addFunction(new Twig_SimpleFunction("mtk_settings_url", function ($moduleId) {
            return mtk_settings_url($moduleId);
        }));
        $twig->addFunction(new Twig_SimpleFunction("mtk_resource_url", function ($section, $name, $locale) {
            return mtk_resource_url($section, $name, $locale, false);
        }));
        $twig->addFunction(new Twig_SimpleFunction("mtk_settings_delete_resource_post_url", function ($section, $name, $locale = "") {
            return mtk_settings_delete_resource_post_url($section, $name, $locale);
        }));

        View::newInstance()->_exportVariableToView("mtk_templating_engine", $twig);
    } else {
        $twig = __get("mtk_templating_engine");
    }

    return $twig->loadTemplate($template . ".twig")->render($data);
}

function mtk_default_item_style($field)
{
    if (Params::getParam("sShowAs") != "list" && Params::getParam("sShowAs") != "gallery") {
        // Get the default item style.
        $defaultItemStyle = osc_default_show_as_at_search();
        if (mtk_field("layout", $field) == 0) {
            $defaultItemStyle = "list";
        } elseif (mtk_field("layout", $field) == 1) {
            $defaultItemStyle = "gallery";
        }

        // Set the param & the view variable.
        Params::setParam("sShowAs", $defaultItemStyle);
        View::newInstance()->_exportVariableToView("search_show_as", $defaultItemStyle);
    }
}

function mkt_follow_rules()
{
    $location = Rewrite::newInstance()->get_location();
    $section  = Rewrite::newInstance()->get_section();

    $result = true;

    $canonical = "";

    switch ($location) {
        case ('item'):
            switch ($section) {
                case 'item_add':
                    $result = true;
                    $canonical = osc_item_post_url();
                    break;
                case 'item_edit':
                    $result = false;
                    $canonical = osc_item_edit_url();
                    break;
                case 'send_friend':
                    $result = false;
                    $canonical = osc_item_send_friend_url();
                    break;
                case 'contact':
                    $result = false;
                    $canonical = osc_item_contact_url();
                    break;
                default:
                    $result = true;
                    $canonical = osc_item_url();
                    break;
            }
            break;
        case ('page'):
            $result = true;
            $canonical = osc_static_page_url();
            break;
        case ('error'):
            $result = false;
            break;
        case ('search'):
            $result = true;
            $params = array(
                'sShowAs'    => null,
                'sOrder'     => null,
                'iOrderType' => null,
                'sPriceMin'  => null,
                'sPriceMax'  => null
            );

            if (Params::getParam('sCity') != "") {
                // If city is set return an url without country and region
                $params['sRegion'] = null;
                $params['sCountry'] = null;
            } elseif (Params::getParam('sRegion') != "") {
                // If region is set return an url without country
                $params['sCountry'] = null;
            }

            if (Params::existParam('sCategory') && is_numeric(Params::getParam('sCategory'))) {
                $category = Category::newInstance()->findByPrimarykey(Params::getParam('sCategory'));
                if ($category) {
                    $params['sCategory'] = $category['s_slug'];
                }
            }
            $canonical = osc_update_search_url($params);
            break;
        case ('login'):
            $result = true;
            $canonical = osc_user_login_url();
            break;
        case ('register'):
            $result = true;
            $canonical = osc_register_account_url();
            break;
        case ('user'):
            switch ($section) {
                case ('pub_profile'):
                    $result = true;
                    $canonical = osc_user_public_profile_url();
                    break;
                default:
                    $result = false;
                    break;
            }
            break;
        case ('contact'):
            $result = true;
            $canonical = osc_contact_url();
            break;
        case (''):
            switch ($section) {
                case (''):
                    $result = true;
                    $canonical = osc_base_url();
                break;
            }
            break;
        default:
            $result = true;
            break;
    }

    View::newInstance()->_exportVariableToView('canonical', $canonical);

    echo "<!-- Robots start -->";
    if ($result) {
        echo '<meta name="robots" content="index, follow" />' . PHP_EOL;
        echo '<meta name="googlebot" content="index, follow" />' . PHP_EOL;
    } else {
        echo '<meta name="robots" content="noindex, nofollow, noarchive" />' . PHP_EOL;
        echo '<meta name="googlebot" content="noindex, nofollow, noarchive" />' . PHP_EOL;
    }
    echo "<!-- Robots end -->";

    if ( osc_get_canonical() != '' ) {
        echo "<!-- Canonical start -->";
        echo '<link rel="canonical" href="' . osc_get_canonical() . '"/>';
        echo "<!-- Canonical end -->";
    }
}

function mkt_max_price() {
    $search = new Search();
    $catId = osc_search_category_id();
    $catId = isset($catId[0]) ? $catId[0] : '';
    $search->addCategory($catId);
    $search->addCountry(osc_search_country());
    $search->addRegion(osc_search_region());
    $search->addCity(osc_search_city());
    $search->order('i_price', 'DESC');
    $search->limit(0, 2);

    $result = $search->doSearch();

    $maxPrice = 0;
    foreach($result as $item) {
        if($maxPrice < $item['i_price']) {
            $maxPrice = $item['i_price'];
        }
    }

    if ($maxPrice == 0 || $maxPrice == "") {
        $maxPrice = (mtk_field("$.layout", "i_search_price_max")) ? mtk_field("$.layout", "i_search_price_max") * 1000000: 1000 * 1000000;
    }

    return ceil($maxPrice/1000000);
}

function mkt_breadcrumb($separator = '&raquo;', $echo = true, $lang = array()) {
    // If madhouse_seo is enabled, use it instead.
    if (osc_plugin_is_enabled("madhouse_seo/index.php")) {
        osc_run_hook('mdh_seo_breadcrumb');
        return;
    }

    // Fallback to Mcfly's breadcrumb.
    $br = new Madhouse_ThemeKitchen_Breadcrumb($lang);
    $br->init();
    if ($echo) {
        echo $br->render($separator);
        return;
    }
    return $br->render($separator);
}

function mtk_title() {
    if (function_exists("mdh_seo_title")) {
        return mdh_seo_title();
    }
    $title = str_replace(' - ' . osc_page_title(), "",  meta_title()); // Use meta title for default page title
    if (osc_item_title() != "") {
        $title = osc_item_title(); // Use meta title for default page title
    }
    return $title;
}
