<?php

//use Peekmo\JsonPath\JsonStore as JsonObject;
use JsonPath\JsonObject as JsonObject;

class Madhouse_ThemeKitchen_Settings2
{
    protected $data;

    public function __construct($data = null, $locales = null)
    {
        if (is_null($data)) {
            $data = array();
        }

        if (is_null($locales)) {
            $locales = array();
        }

        $this->data = $data;
        $this->locales = $locales;
    }

    public function normalizeQuery($query)
    {
        // Cleanup.
        $query = array_values(
            array_filter(
                $query,
                function ($queryPart) {
                    // Remove empty strings.
                    return ($queryPart !== "");
                }
            )
        );

        // Iterate over each part of the query.
        for ($i = 0; $i < count($query); ++$i) {
            $isIndex = preg_match('/^\d+$/', $query[$i]);
            if ($isIndex) {
                // Add square brackets for ints.
                $query[$i] = "[" . $query[$i] . "]";
            } else {
                // Add quote if needed.
                // [something-with-dash] -> ["something-with-dash"]
                $query[$i] = preg_replace(
                    '/(\.|\[)?(\w+[\-\$\s][^\.\]]+)(\.|\])?/',
                    '["$2"]',
                    $query[$i]
                );
            }

            // Add "$" if missing at root.
            if ($i === 0 && !preg_match('/^\$(.*)/', $query[0])) {
                if (preg_match('/^\[\d+\]$/', $query[0])) {
                    // $[3].s_image
                    $query[0] = "$" . $query[0];
                } else {
                    // $.general.s_title
                    $query[0] = "$." . $query[0];
                }
            } else {
                // $..[3] -> $..*[3]
                if ($isIndex && isset($query[$i - 1]) && $query[$i - 1] === "") {
                    $query[$i] = "*" . $query[$i];
                }
            }
        }

        // Always return a valid query.
        if (empty($query)) {
            $query[] = "$";
        }

        // Return as array.
        return $query;
    }

    /**
     * Build the query with optional $field name and $locale locale code.
     *
     * @param  string|array $section
     * @param  string $field
     * @param  string $locale
     *
     * @return array
     */
    public function buildQuery($section, $field = null, $locale = "")
    {
        // Start the query.
        $queryFull = (is_array($section)) ? $section : array($section);

        if (isset($locale) && $locale !== "") {
            // Add locale if requested.
            array_push($queryFull, "locale", $locale);
        }

        if (!is_null($field) && !empty($field)) {
            // Add field if requested.
            array_push($queryFull, $field);
        }

        return $queryFull;
    }

    /**
     * Serialize the query.
     *
     * @param  array $query
     *
     * @return string
     */
    public function serializeQuery($query, $normalize = true)
    {
        if ($normalize) {
            // Cleanup and normalize the query with jsonpath requirements.
            $query = $this->normalizeQuery($query);
        }

        // Implode with dot.
        $queryText = implode(".", $query);

        // $.something.[3] -> $.something[3]
        return preg_replace('/\.\[/', '[', $queryText);
    }

    /**
     * Check if the $query contains jsonpath filter expr.
     *
     * Ex. $.categories[?(@.b_highlighted=='1')].pk_i_id
     *
     * @param  string|array  $query
     *
     * @return boolean
     */
    public function hasFilter($query)
    {
        $filters = array_filter(
            is_array($query) ? $query : array($query),
            function ($expr) {
                return preg_match('/^.*\[\?\(.*\)\].*$/', $expr);
            }
        );

        return count($filters);
    }

    public function set($expr, $value)
    {
        if ($expr === "$") {
            return;
        }

        // JsonStore enables us to make JSONPath query on $settings.
        $store = new JsonObject($this->data);

        // Normalize and concat.
        $expr = $this->serializeQuery((is_array($expr)) ? $expr : array($expr));

        if ($store->get($expr)) {
            // Execute query on settings, set the new value.
            $res = @$store->set($expr, $value);

            $this->data = $store->getValue();
        } else {
            $parts = explode(".", $expr);
            $field = array_pop($parts);

            if (count($parts) === 1 && $parts[0] === "$") {
                // Don't use JSONPath, don't work with root...
                $this->data[$field] = $value;
            } else {
                // Just add the value to parent.
                $res = @$store->add($this->serializeQuery($parts), $value, $field);

                // Get the updated array from the store.
                $this->data = $store->getValue();
            }
        }
    }

    /**
     * Get a $field from $section (for a $locale) in settings.
     * @param $expr     section as a JSONPath expression.
     * @param $field    settings field.
     * @param $locale   locale to retrieve.
     * @return Any
     * @since 1.00
     */
    public function get($section = "$", $field = null, $locale = "", $fallback = true)
    {
        // Special case where we want the whole settings object.
        if ($section === "$" && is_null($field)) {
            return $this->data;
        }

        // JSON query object.
        $store = new JsonObject($this->data);

        // Build the query.
        $queryFull = $this->buildQuery($section, $field, $locale);

        // Execute query on settings & retrieve results (if any).
        $res = $store->get($this->serializeQuery($queryFull));

        // Take the first result only.
        if ($res) {
            $res2 = (!$this->hasFilter($queryFull)) ? $res[0] : $res;
        } else {
            $res2 = (!is_null($field)) ? "" : array();
        }

        // Deals with empty results.
        if ($res2 === "") {
            if ($fallback === false) {
                // Don't fallback, no results found.
                return "";
            } else {
                // Try to find an available language.
                foreach (osc_get_locales() as $l) {
                    // Build a new query for that locale.
                    $localizedQuery = $this->buildQuery($section, $field, $l["pk_c_code"]);

                    // Perform the query to find a fallback result.
                    $localeResults = $store->get($this->serializeQuery($localizedQuery));
                    if ($localeResults !== false && $localeResults[0] !== "") {
                        // One locale is defined, return that as fallback value.
                        return array_shift($localeResults);
                    }
                }


                // No locale available anywhere for $field.
                return "";
            }
        }

        return $res2;
    }

    public function getSubset($section)
    {
        return new self($this->get($section));
    }

    public function isEmpty()
    {
        // If null or empty.
        if (is_null($this->data) || empty($this->data)) {
            return true;
        }

        // Check if one module has settings.
        foreach ($this->data as $moduleId => $moduleSettings) {
            if (!empty($moduleSettings)) {
                return false;
            }
        }

        // Default: empty.
        return true;
    }

    public function toArray()
    {
        return $this->data;
    }
}
